<?php
defined( 'ABSPATH' ) || exit;

class HRM_Policies {

    public static function init() {
        // Admin AJAX handlers
        add_action( 'wp_ajax_hrm_save_policy', [ __CLASS__, 'handle_ajax_save_policy' ] );
        add_action( 'wp_ajax_hrm_delete_policy', [ __CLASS__, 'handle_ajax_delete_policy' ] );
        add_action( 'wp_ajax_hrm_get_policy', [ __CLASS__, 'handle_ajax_get_policy' ] );
        
        // Employee AJAX handlers
        add_action( 'wp_ajax_hrm_get_policies', [ __CLASS__, 'handle_ajax_get_policies' ] );
        add_action( 'wp_ajax_nopriv_hrm_get_policies', [ __CLASS__, 'handle_ajax_get_policies' ] );
    }

    /**
     * Render Admin Policies Page
     */
    public static function render_admin_policies() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_policies';

        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name;
        
        // Get filter parameters
        $category_filter = isset($_GET['category']) ? sanitize_text_field($_GET['category']) : '';
        $status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';

        // Pagination
        $per_page = HRM_Pagination::get_per_page();
        $current_page = HRM_Pagination::get_current_page();
        $offset = HRM_Pagination::get_offset($current_page, $per_page);

        // Build query
        $where = ['1=1'];
        $params = [];

        if ($category_filter) {
            $where[] = "category = %s";
            $params[] = $category_filter;
        }

        if ($status_filter) {
            $where[] = "status = %s";
            $params[] = $status_filter;
        }

        $where_clause = implode(' AND ', $where);

        // Get policies
        $policies = [];
        $total_policies = 0;
        $stats = ['total' => 0, 'active' => 0, 'draft' => 0, 'archived' => 0];

        if ($table_exists) {
            if (!empty($params)) {
                $total_query = "SELECT COUNT(*) FROM $table_name WHERE $where_clause";
                $total_policies = $wpdb->get_var($wpdb->prepare($total_query, $params));
                
                $query = "SELECT * FROM $table_name WHERE $where_clause ORDER BY created_at DESC LIMIT %d OFFSET %d";
                $params[] = $per_page;
                $params[] = $offset;
                $policies = $wpdb->get_results($wpdb->prepare($query, $params));
            } else {
                $total_policies = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE $where_clause");
                $policies = $wpdb->get_results($wpdb->prepare(
                    "SELECT * FROM $table_name WHERE $where_clause ORDER BY created_at DESC LIMIT %d OFFSET %d",
                    $per_page, $offset
                ));
            }

            $stats = [
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name"),
                'active' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'active'"),
                'draft' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'draft'"),
                'archived' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'archived'")
            ];
        }

        $total_pages = ceil($total_policies / $per_page);

        // Categories
        $categories = ['general', 'hr', 'it', 'finance', 'safety', 'compliance', 'other'];
        ?>
        <div class="hrm-wrap">
            <!-- Header -->
            <div class="hrm-header-section">
                <div class="hrm-header-left">
                    <h1>Company Policies</h1>
                    <p>Manage and distribute company policies to employees</p>
                </div>
                <div class="hrm-header-right">
                    <button class="hrm-btn-primary" id="hrm-add-policy-btn">
                        <span class="dashicons dashicons-plus-alt2"></span> Add Policy
                    </button>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="hrm-stats-row">
                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Total Policies</span>
                        <span class="dashicons dashicons-media-document"></span>
                    </div>
                    <div class="stat-number"><?php echo esc_html($stats['total']); ?></div>
                    <div class="stat-footer">All policies</div>
                </div>

                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Active</span>
                        <span class="dashicons dashicons-yes-alt"></span>
                    </div>
                    <div class="stat-number" style="color: #10B981;"><?php echo esc_html($stats['active']); ?></div>
                    <div class="stat-footer">Visible to employees</div>
                </div>

                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Drafts</span>
                        <span class="dashicons dashicons-edit"></span>
                    </div>
                    <div class="stat-number" style="color: #F59E0B;"><?php echo esc_html($stats['draft']); ?></div>
                    <div class="stat-footer">Work in progress</div>
                </div>

                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Archived</span>
                        <span class="dashicons dashicons-archive"></span>
                    </div>
                    <div class="stat-number" style="color: #6B7280;"><?php echo esc_html($stats['archived']); ?></div>
                    <div class="stat-footer">No longer active</div>
                </div>
            </div>

            <!-- Filters -->
            <div class="hrm-card">
                <div class="card-header">
                    <h3>Filter Policies</h3>
                </div>
                <div class="hrm-filter-bar" style="padding: 4px 0;">
                    <form method="get" style="display: flex; gap: 10px; align-items: center; flex-wrap: nowrap; width: 100%;">
                        <input type="hidden" name="page" value="hrm-policies">
                        
                        <select name="category" class="hrm-select-input" style="flex: 1 1 0; min-width: 0;">
                            <option value="">All Categories</option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo esc_attr($cat); ?>" <?php selected($category_filter, $cat); ?>>
                                    <?php echo esc_html(ucfirst($cat)); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>

                        <select name="status" class="hrm-select-input" style="flex: 1 1 0; min-width: 0;">
                            <option value="">All Status</option>
                            <option value="active" <?php selected($status_filter, 'active'); ?>>Active</option>
                            <option value="draft" <?php selected($status_filter, 'draft'); ?>>Draft</option>
                            <option value="archived" <?php selected($status_filter, 'archived'); ?>>Archived</option>
                        </select>

                        <button type="submit" class="hrm-btn-primary" style="display: flex; align-items: center; gap: 6px; white-space: nowrap; flex-shrink: 0; padding: 10px 20px;">
                            <span class="dashicons dashicons-search" style="font-size: 18px; line-height: 1;"></span>
                            <span>Apply</span>
                        </button>
                    </form>
                </div>
            </div>

            <!-- Policies Table -->
            <div class="hrm-card">
                <div class="card-header">
                    <h3>Policies (<?php echo esc_html($total_policies); ?>)</h3>
                </div>

                <?php if (empty($policies)): ?>
                    <div style="padding: 60px 20px; text-align: center; color: #6B7280;">
                        <span class="dashicons dashicons-media-document" style="font-size: 64px; opacity: 0.3;"></span>
                        <p style="margin-top: 16px; font-size: 16px;">No policies found</p>
                        <button class="hrm-btn-primary" onclick="document.getElementById('hrm-add-policy-btn').click()" style="margin-top: 16px;">
                            <span class="dashicons dashicons-plus-alt2"></span> Add Your First Policy
                        </button>
                    </div>
                <?php else: ?>
                    <div class="hrm-table-container" style="overflow-x: auto;">
                        <table class="hrm-table">
                            <thead>
                                <tr>
                                    <th style="min-width: 200px;">Policy Title</th>
                                    <th style="min-width: 100px;">Category</th>
                                    <th style="min-width: 120px;">Effective Date</th>
                                    <th style="min-width: 90px;">Status</th>
                                    <th style="min-width: 80px;">Document</th>
                                    <th style="min-width: 100px; text-align: center;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($policies as $policy): 
                                    $status_classes = [
                                        'active' => 'hrm-badge-success',
                                        'draft' => 'hrm-badge-warning',
                                        'archived' => 'hrm-badge-default'
                                    ];
                                    $badge_class = isset($status_classes[$policy->status]) ? $status_classes[$policy->status] : 'hrm-badge-default';
                                ?>
                                <tr class="hrm-policy-row" data-policy-id="<?php echo esc_attr($policy->id); ?>">
                                    <td>
                                        <div class="hrm-policy-title-cell">
                                            <div class="hrm-policy-icon-wrapper">
                                                <span class="dashicons dashicons-media-document"></span>
                                            </div>
                                            <div>
                                                <div class="hrm-policy-title"><?php echo esc_html($policy->title); ?></div>
                                                <div class="hrm-policy-excerpt">
                                                    <?php echo esc_html(wp_trim_words($policy->description, 10, '...')); ?>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="hrm-category-badge hrm-category-<?php echo esc_attr($policy->category); ?>">
                                            <?php echo esc_html(ucfirst($policy->category)); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="hrm-policy-date">
                                            <?php echo esc_html(date('M d, Y', strtotime($policy->effective_date))); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="hrm-badge <?php echo esc_attr($badge_class); ?>">
                                            <?php echo esc_html(strtoupper($policy->status)); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($policy->attachment_url): ?>
                                            <a href="<?php echo esc_url($policy->attachment_url); ?>" target="_blank" class="hrm-btn-icon hrm-download-icon" title="Download Document">
                                                <span class="dashicons dashicons-download"></span>
                                            </a>
                                        <?php else: ?>
                                            <span class="hrm-text-muted">—</span>
                                        <?php endif; ?>
                                    </td>
                                    <td style="text-align: center;">
                                        <div style="display: inline-flex; gap: 6px;">
                                            <button class="hrm-action-btn hrm-edit-policy" 
                                                    data-policy-id="<?php echo esc_attr($policy->id); ?>"
                                                    title="Edit">
                                                <span class="dashicons dashicons-edit"></span>
                                            </button>
                                            <button class="hrm-action-btn hrm-action-reject hrm-delete-policy" 
                                                    data-policy-id="<?php echo esc_attr($policy->id); ?>"
                                                    title="Delete">
                                                <span class="dashicons dashicons-trash"></span>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    <?php 
                    HRM_Pagination::render([
                        'total_items' => $total_policies,
                        'current_page' => $current_page,
                        'per_page' => $per_page,
                        'query_args' => [
                            'page' => 'hrm-policies',
                            'category' => $category_filter,
                            'status' => $status_filter
                        ],
                        'show_sizes' => true,
                        'context' => 'admin'
                    ]);
                    HRM_Pagination::render_script();
                    ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Add/Edit Policy Modal -->
        <div id="hrm-policy-modal" class="hrm-modal" style="display: none;">
            <div class="hrm-modal-content hrm-premium-modal" style="max-width: 650px;">
                <div class="hrm-modal-header-premium">
                    <div class="header-icon-wrapper" style="background: rgba(12, 135, 118, 0.15); color: #0C8776;">
                        <span class="dashicons dashicons-media-document"></span>
                    </div>
                    <div class="header-text">
                        <h2 id="policy-modal-title">Add New Policy</h2>
                        <p>Create or edit company policy</p>
                    </div>
                    <button class="hrm-modal-close-premium" onclick="closePolicyModal()">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                
                <div class="hrm-modal-body-premium" style="padding: 24px;">
                    <form id="hrm-policy-form" enctype="multipart/form-data">
                        <input type="hidden" id="policy-id" name="policy_id" value="">
                        
                        <div class="hrm-form-row">
                            <div class="hrm-form-group" style="flex: 2;">
                                <label for="policy-title">Policy Title <span style="color: #EF4444;">*</span></label>
                                <input type="text" id="policy-title" name="title" class="hrm-input" required placeholder="e.g., Remote Work Policy">
                            </div>
                            <div class="hrm-form-group" style="flex: 1;">
                                <label for="policy-category">Category <span style="color: #EF4444;">*</span></label>
                                <select id="policy-category" name="category" class="hrm-input" required>
                                    <option value="">Select Category</option>
                                    <?php foreach ($categories as $cat): ?>
                                        <option value="<?php echo esc_attr($cat); ?>"><?php echo esc_html(ucfirst($cat)); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="hrm-form-group">
                            <label for="policy-description">Description <span style="color: #EF4444;">*</span></label>
                            <textarea id="policy-description" name="description" class="hrm-input" rows="5" required placeholder="Describe the policy details..."></textarea>
                        </div>

                        <div class="hrm-form-row">
                            <div class="hrm-form-group">
                                <label for="policy-effective-date">Effective Date <span style="color: #EF4444;">*</span></label>
                                <input type="date" id="policy-effective-date" name="effective_date" class="hrm-input" required>
                            </div>
                            <div class="hrm-form-group">
                                <label for="policy-status">Status <span style="color: #EF4444;">*</span></label>
                                <select id="policy-status" name="status" class="hrm-input" required>
                                    <option value="draft">Draft (Hidden)</option>
                                    <option value="active">Active (Visible to Employees)</option>
                                    <option value="archived">Archived</option>
                                </select>
                            </div>
                        </div>

                        <div class="hrm-form-group">
                            <label for="policy-attachment">Policy Document (PDF/DOC)</label>
                            <div class="hrm-file-upload-area" id="policy-upload-area">
                                <input type="file" id="policy-attachment" name="attachment" accept=".pdf,.doc,.docx" style="display: none;">
                                <div class="upload-placeholder" id="upload-placeholder">
                                    <span class="dashicons dashicons-cloud-upload" style="font-size: 32px; color: #9CA3AF;"></span>
                                    <p style="margin: 8px 0 0; color: #6B7280;">Click to upload or drag and drop</p>
                                    <span style="font-size: 12px; color: #9CA3AF;">PDF, DOC, DOCX (Max 10MB)</span>
                                </div>
                                <div class="upload-preview" id="upload-preview" style="display: none;">
                                    <span class="dashicons dashicons-media-document" style="font-size: 24px; color: #0C8776;"></span>
                                    <span id="upload-filename" style="font-weight: 500; color: #374151;"></span>
                                    <button type="button" id="remove-attachment" class="hrm-btn-icon" style="margin-left: auto;">
                                        <span class="dashicons dashicons-no-alt" style="color: #EF4444;"></span>
                                    </button>
                                </div>
                            </div>
                            <div id="current-attachment" style="margin-top: 8px; display: none;">
                                <span style="font-size: 13px; color: #6B7280;">Current: </span>
                                <a id="current-attachment-link" href="#" target="_blank" style="font-size: 13px; color: #0C8776;">View Document</a>
                            </div>
                        </div>

                        <div class="hrm-form-actions" style="display: flex; gap: 12px; justify-content: flex-end; margin-top: 24px; padding-top: 20px; border-top: 1px solid #E5E7EB;">
                            <button type="button" class="hrm-btn-outline" onclick="closePolicyModal()">Cancel</button>
                            <button type="submit" class="hrm-btn-primary" id="save-policy-btn">
                                <span class="dashicons dashicons-saved"></span> Save Policy
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Delete Confirmation Modal -->
        <div id="hrm-delete-policy-modal" class="hrm-modal" style="display: none;">
            <div class="hrm-modal-content" style="max-width: 400px; text-align: center; padding: 32px;">
                <div style="width: 64px; height: 64px; background: #FEE2E2; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
                    <span class="dashicons dashicons-warning" style="font-size: 32px; color: #EF4444;"></span>
                </div>
                <h3 style="margin: 0 0 12px; font-size: 18px; color: #111827;">Delete Policy?</h3>
                <p style="margin: 0 0 24px; color: #6B7280;">This action cannot be undone. The policy will be permanently removed.</p>
                <input type="hidden" id="delete-policy-id" value="">
                <div style="display: flex; gap: 12px; justify-content: center;">
                    <button class="hrm-btn-outline" onclick="closeDeleteModal()">Cancel</button>
                    <button class="hrm-btn-red" id="confirm-delete-btn">
                        <span class="dashicons dashicons-trash"></span> Delete
                    </button>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            
            // Open Add Policy Modal
            $('#hrm-add-policy-btn').on('click', function() {
                $('#policy-modal-title').text('Add New Policy');
                $('#hrm-policy-form')[0].reset();
                $('#policy-id').val('');
                $('#upload-placeholder').show();
                $('#upload-preview').hide();
                $('#current-attachment').hide();
                $('body').addClass('hrm-modal-open');
                $('#hrm-policy-modal').fadeIn(200);
            });

            // Close Policy Modal
            window.closePolicyModal = function() {
                $('#hrm-policy-modal').fadeOut(200);
                $('body').removeClass('hrm-modal-open');
            };

            // Close Delete Modal
            window.closeDeleteModal = function() {
                $('#hrm-delete-policy-modal').fadeOut(200);
                $('body').removeClass('hrm-modal-open');
            };

            // File Upload Area
            const uploadArea = $('#policy-upload-area');
            const fileInput = $('#policy-attachment');

            uploadArea.on('click', function() {
                fileInput.trigger('click');
            });

            uploadArea.on('dragover', function(e) {
                e.preventDefault();
                $(this).addClass('dragover');
            });

            uploadArea.on('dragleave', function() {
                $(this).removeClass('dragover');
            });

            uploadArea.on('drop', function(e) {
                e.preventDefault();
                $(this).removeClass('dragover');
                const files = e.originalEvent.dataTransfer.files;
                if (files.length) {
                    fileInput[0].files = files;
                    showFilePreview(files[0]);
                }
            });

            fileInput.on('change', function() {
                if (this.files.length) {
                    showFilePreview(this.files[0]);
                }
            });

            function showFilePreview(file) {
                $('#upload-filename').text(file.name);
                $('#upload-placeholder').hide();
                $('#upload-preview').show();
            }

            $('#remove-attachment').on('click', function(e) {
                e.stopPropagation();
                fileInput.val('');
                $('#upload-placeholder').show();
                $('#upload-preview').hide();
            });

            // Edit Policy
            $('.hrm-edit-policy').on('click', function() {
                const policyId = $(this).data('policy-id');
                
                $.ajax({
                    url: hrm_admin_params.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'hrm_get_policy',
                        nonce: hrm_admin_params.nonce,
                        policy_id: policyId
                    },
                    success: function(response) {
                        if (response.success) {
                            const policy = response.data;
                            $('#policy-modal-title').text('Edit Policy');
                            $('#policy-id').val(policy.id);
                            $('#policy-title').val(policy.title);
                            $('#policy-category').val(policy.category);
                            $('#policy-description').val(policy.description);
                            $('#policy-effective-date').val(policy.effective_date);
                            $('#policy-status').val(policy.status);
                            
                            if (policy.attachment_url) {
                                $('#current-attachment-link').attr('href', policy.attachment_url);
                                $('#current-attachment').show();
                            } else {
                                $('#current-attachment').hide();
                            }
                            
                            $('#upload-placeholder').show();
                            $('#upload-preview').hide();
                            $('body').addClass('hrm-modal-open');
                            $('#hrm-policy-modal').fadeIn(200);
                        }
                    }
                });
            });

            // Delete Policy
            $('.hrm-delete-policy').on('click', function() {
                const policyId = $(this).data('policy-id');
                $('#delete-policy-id').val(policyId);
                $('body').addClass('hrm-modal-open');
                $('#hrm-delete-policy-modal').fadeIn(200);
            });

            // Confirm Delete
            $('#confirm-delete-btn').on('click', function() {
                const policyId = $('#delete-policy-id').val();
                const btn = $(this);
                btn.prop('disabled', true).html('<span class="dashicons dashicons-update hrm-spin"></span> Deleting...');

                $.ajax({
                    url: hrm_admin_params.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'hrm_delete_policy',
                        nonce: hrm_admin_params.nonce,
                        policy_id: policyId
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || 'Error deleting policy');
                            btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Delete');
                        }
                    },
                    error: function() {
                        alert('Server error. Please try again.');
                        btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Delete');
                    }
                });
            });

            // Save Policy
            $('#hrm-policy-form').on('submit', function(e) {
                e.preventDefault();
                
                const formData = new FormData(this);
                formData.append('action', 'hrm_save_policy');
                formData.append('nonce', hrm_admin_params.nonce);

                const btn = $('#save-policy-btn');
                btn.prop('disabled', true).html('<span class="dashicons dashicons-update hrm-spin"></span> Saving...');

                $.ajax({
                    url: hrm_admin_params.ajaxurl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || 'Error saving policy');
                            btn.prop('disabled', false).html('<span class="dashicons dashicons-saved"></span> Save Policy');
                        }
                    },
                    error: function() {
                        alert('Server error. Please try again.');
                        btn.prop('disabled', false).html('<span class="dashicons dashicons-saved"></span> Save Policy');
                    }
                });
            });

            // Close modals on overlay click
            $('.hrm-modal').on('click', function(e) {
                if ($(e.target).is('.hrm-modal')) {
                    $(this).fadeOut(200);
                    $('body').removeClass('hrm-modal-open');
                }
            });

            // Close on escape key
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    $('.hrm-modal:visible').fadeOut(200);
                    $('body').removeClass('hrm-modal-open');
                }
            });
        });
        </script>
        <?php
    }

    /**
     * AJAX: Save Policy
     */
    public static function handle_ajax_save_policy() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized access']);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_policies';

        $policy_id = isset($_POST['policy_id']) ? intval($_POST['policy_id']) : 0;
        $title = sanitize_text_field($_POST['title']);
        $category = sanitize_text_field($_POST['category']);
        $description = sanitize_textarea_field($_POST['description']);
        $effective_date = sanitize_text_field($_POST['effective_date']);
        $status = sanitize_text_field($_POST['status']);

        if (empty($title) || empty($category) || empty($description) || empty($effective_date)) {
            wp_send_json_error(['message' => 'Please fill in all required fields']);
        }

        // Handle file upload
        $attachment_url = '';
        $attachment_id = 0;
        
        if (!empty($_FILES['attachment']['name'])) {
            $allowed_types = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
            
            if (!in_array($_FILES['attachment']['type'], $allowed_types)) {
                wp_send_json_error(['message' => 'Invalid file type. Only PDF and DOC files are allowed.']);
            }

            if ($_FILES['attachment']['size'] > 10 * 1024 * 1024) {
                wp_send_json_error(['message' => 'File size must be less than 10MB']);
            }

            require_once(ABSPATH . 'wp-admin/includes/file.php');
            require_once(ABSPATH . 'wp-admin/includes/media.php');
            require_once(ABSPATH . 'wp-admin/includes/image.php');

            $attachment_id = media_handle_upload('attachment', 0);
            
            if (is_wp_error($attachment_id)) {
                wp_send_json_error(['message' => 'Error uploading file: ' . $attachment_id->get_error_message()]);
            }

            $attachment_url = wp_get_attachment_url($attachment_id);
        }

        $data = [
            'title' => $title,
            'category' => $category,
            'description' => $description,
            'effective_date' => $effective_date,
            'status' => $status,
            'updated_at' => current_time('mysql')
        ];

        if ($attachment_url) {
            $data['attachment_url'] = $attachment_url;
            $data['attachment_id'] = $attachment_id;
        }

        if ($policy_id > 0) {
            // Update existing
            $wpdb->update($table_name, $data, ['id' => $policy_id]);
        } else {
            // Insert new
            $data['created_at'] = current_time('mysql');
            $data['created_by'] = get_current_user_id();
            $wpdb->insert($table_name, $data);
        }

        wp_send_json_success(['message' => 'Policy saved successfully']);
    }

    /**
     * AJAX: Delete Policy
     */
    public static function handle_ajax_delete_policy() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized access']);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_policies';
        $policy_id = intval($_POST['policy_id']);

        // Delete attachment if exists
        $policy = $wpdb->get_row($wpdb->prepare("SELECT attachment_id FROM $table_name WHERE id = %d", $policy_id));
        if ($policy && $policy->attachment_id) {
            wp_delete_attachment($policy->attachment_id, true);
        }

        $wpdb->delete($table_name, ['id' => $policy_id]);

        wp_send_json_success(['message' => 'Policy deleted successfully']);
    }

    /**
     * AJAX: Get Policy
     */
    public static function handle_ajax_get_policy() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized access']);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_policies';
        $policy_id = intval($_POST['policy_id']);

        $policy = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $policy_id));

        if ($policy) {
            wp_send_json_success($policy);
        } else {
            wp_send_json_error(['message' => 'Policy not found']);
        }
    }

    /**
     * AJAX: Get Policies (for employees)
     */
    public static function handle_ajax_get_policies() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        
        if (!is_user_logged_in()) {
            wp_send_json_error(['message' => 'Please log in to view policies']);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_policies';

        $policies = $wpdb->get_results(
            "SELECT id, title, category, description, effective_date, attachment_url 
             FROM $table_name 
             WHERE status = 'active' 
             ORDER BY effective_date DESC"
        );

        wp_send_json_success($policies);
    }

    /**
     * Render Employee Policies View
     */
    public static function render_employee_policies() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_policies';

        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name;

        $policies = [];
        if ($table_exists) {
            $policies = $wpdb->get_results(
                "SELECT * FROM $table_name WHERE status = 'active' ORDER BY effective_date DESC"
            );
        }

        $categories = ['general', 'hr', 'it', 'finance', 'safety', 'compliance', 'other'];
        ?>
        <div class="hrm-policies-section">
            <div class="hrm-section-header">
                <h2><span class="dashicons dashicons-media-document"></span> Company Policies</h2>
            </div>

            <!-- Category Filter -->
            <div class="hrm-policy-filters" style="margin-bottom: 24px;">
                <div class="hrm-filter-chips">
                    <button class="hrm-filter-chip active" data-category="all">All</button>
                    <?php foreach ($categories as $cat): ?>
                        <button class="hrm-filter-chip" data-category="<?php echo esc_attr($cat); ?>">
                            <?php echo esc_html(ucfirst($cat)); ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>

            <?php if (empty($policies)): ?>
                <div class="hrm-empty-state-modern">
                    <div class="empty-icon">
                        <span class="dashicons dashicons-media-document"></span>
                    </div>
                    <h4>No Policies Available</h4>
                    <p>There are no company policies published at this time.</p>
                </div>
            <?php else: ?>
                <div class="hrm-policies-grid" id="hrm-policies-grid">
                    <?php foreach ($policies as $policy): ?>
                        <div class="hrm-policy-card" data-category="<?php echo esc_attr($policy->category); ?>">
                            <div class="policy-card-header">
                                <div class="policy-icon">
                                    <span class="dashicons dashicons-media-document"></span>
                                </div>
                                <span class="hrm-category-badge hrm-category-<?php echo esc_attr($policy->category); ?>">
                                    <?php echo esc_html(ucfirst($policy->category)); ?>
                                </span>
                            </div>
                            <div class="policy-card-body">
                                <h3 class="policy-title"><?php echo esc_html($policy->title); ?></h3>
                                <p class="policy-description"><?php echo esc_html(wp_trim_words($policy->description, 20, '...')); ?></p>
                                <div class="policy-meta">
                                    <span class="dashicons dashicons-calendar-alt"></span>
                                    Effective: <?php echo esc_html(date('M d, Y', strtotime($policy->effective_date))); ?>
                                </div>
                            </div>
                            <div class="policy-card-footer">
                                <button class="hrm-btn-sm hrm-btn-outline-primary hrm-view-policy" 
                                        data-policy-id="<?php echo esc_attr($policy->id); ?>"
                                        data-title="<?php echo esc_attr($policy->title); ?>"
                                        data-category="<?php echo esc_attr($policy->category); ?>"
                                        data-description="<?php echo esc_attr($policy->description); ?>"
                                        data-effective="<?php echo esc_attr(date('F d, Y', strtotime($policy->effective_date))); ?>"
                                        data-attachment="<?php echo esc_attr($policy->attachment_url); ?>">
                                    <span class="dashicons dashicons-visibility"></span> View Details
                                </button>
                                <?php if ($policy->attachment_url): ?>
                                    <a href="<?php echo esc_url($policy->attachment_url); ?>" target="_blank" class="hrm-btn-sm hrm-btn-primary">
                                        <span class="dashicons dashicons-download"></span> Download
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- Policy Detail Modal -->
        <div id="hrm-policy-detail-modal" class="hrm-modal hrm-frontend-modal" style="display: none;">
            <div class="hrm-modal-content hrm-premium-modal" style="max-width: 600px;">
                <div class="hrm-modal-header-premium">
                    <div class="header-icon-wrapper">
                        <span class="dashicons dashicons-media-document"></span>
                    </div>
                    <div class="header-text">
                        <h2 id="policy-detail-title">Policy Title</h2>
                        <p><span id="policy-detail-category" class="hrm-category-badge"></span></p>
                    </div>
                    <button class="hrm-modal-close-premium" type="button">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                
                <div class="hrm-modal-body-premium">
                    <div class="policy-detail-meta">
                        <span class="dashicons dashicons-calendar-alt"></span>
                        <span class="meta-label">Effective Date: </span>
                        <span id="policy-detail-effective" class="meta-value"></span>
                    </div>
                    
                    <div class="policy-detail-content">
                        <label class="section-label">Policy Details</label>
                        <div id="policy-detail-description" class="policy-description-box">
                        </div>
                    </div>
                    
                    <div id="policy-detail-attachment-section" style="display: none;">
                        <a id="policy-detail-download" href="#" target="_blank" class="hrm-btn-primary" style="width: 100%; justify-content: center;">
                            <span class="dashicons dashicons-download"></span> Download Policy Document
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Filter by category
            $('.hrm-filter-chip').on('click', function() {
                const category = $(this).data('category');
                
                $('.hrm-filter-chip').removeClass('active');
                $(this).addClass('active');
                
                if (category === 'all') {
                    $('.hrm-policy-card').show();
                } else {
                    $('.hrm-policy-card').hide();
                    $('.hrm-policy-card[data-category="' + category + '"]').show();
                }
            });

            // View policy details - Open modal with clean focused layout
            $('.hrm-view-policy').on('click', function() {
                const title = $(this).data('title');
                const category = $(this).data('category');
                const description = $(this).data('description');
                const effective = $(this).data('effective');
                const attachment = $(this).data('attachment');

                $('#policy-detail-title').text(title);
                $('#policy-detail-category').text(category.charAt(0).toUpperCase() + category.slice(1))
                    .removeClass().addClass('hrm-category-badge hrm-category-' + category);
                $('#policy-detail-effective').text(effective);
                $('#policy-detail-description').text(description);

                if (attachment) {
                    $('#policy-detail-download').attr('href', attachment);
                    $('#policy-detail-attachment-section').show();
                } else {
                    $('#policy-detail-attachment-section').hide();
                }

                // Add body class to hide headers/navigation for clean focused layout
                $('body').addClass('hrm-modal-open');
                $('#hrm-policy-detail-modal').fadeIn(200);
            });

            // Close policy detail modal function
            function closePolicyDetailModal() {
                $('#hrm-policy-detail-modal').fadeOut(200);
                $('body').removeClass('hrm-modal-open');
            }

            // Close modal on overlay click
            $('#hrm-policy-detail-modal').on('click', function(e) {
                if ($(e.target).is('.hrm-modal')) {
                    closePolicyDetailModal();
                }
            });

            // Close modal on close button click
            $('#hrm-policy-detail-modal .hrm-modal-close-premium').on('click', function() {
                closePolicyDetailModal();
            });

            // Close modal on escape key
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && $('#hrm-policy-detail-modal').is(':visible')) {
                    closePolicyDetailModal();
                }
            });
        });
        </script>
        <?php
    }
}
